%**************************************************************************
% Solve the pdetool FEM linear system  
%**************************************************************************
%
% DESCRIPTION
% Assemble and solve the pdetool finite element linear system 
% 
% INPUT
% psi:     level-set function
% mesh:    pdetool mesh struct
% matprop: material properties struct
% pdecoef: pdetool coeficiets struct
% bc:      boundary conditions struct
%
% OUTPUT
% [U,F]: pdetool displacement and load vectors
%
% HISTORY
% A.A. Novotny     01/2013: code implementation
% A.A. Novotny     01/2018: code updating
%
%**************************************************************************

function [U,F] = pdesolve(psi, mesh, matprop, pdecoef, bc)

    % mesh parameters
    p = mesh.p; t = mesh.t; np = size(p,2); 
    
    % material properties
    gamma = matprop.gamma; % contrast  
    tgamma = pdeintrp(p,t,(psi<0) + gamma*(psi>=0)); 

    % pde coefficients    
    f0 = zeros(2,1); a0 = pdecoef.a; c0 = pdecoef.c;
    
    % build stiffness metrix and load vector
    c = c0 * tgamma; % effective elasticity tensor 
    [K,M,F] = assema(p,t,c,a0,f0); K = K + M;

    % apply Neumann boundary condition
    if ~isempty(bc.pNeu)
        eq = bc.pNeu(:,1) + (bc.pNeu(:,2)-1)*np; 
        F(eq) = F(eq) + bc.pNeu(:,3); 
    end
    
    % apply Dirichlet boundary condition
    if ~isempty(bc.pDir)
        eq = bc.pDir(:,1) + (bc.pDir(:,2)-1)*np; neq = size(eq,1);
        U = zeros(size(F,1),1); U(eq) = bc.pDir(:,3);
        F = F - K * U; F(eq) = bc.pDir(:,3);
        K(eq,:) = 0.0; K(:,eq) = 0.0; 
        K(eq,eq) = speye(neq);
    end;
    
    % solve linear system
    U = K \ F;   
          
end

