%**************************************************************************
% Topological Derivative for the Structural Compliance Shape Functional
%**************************************************************************
%
% DESCRIPTION
% Computes the topological derivative of the structural compliance
% 
% INPUT
% U:       pdetool solution
% psi:     level-set function
% mesh:    pdetool mesh struct
% matprop: material properties struct
%
% OUTPUT
% dt = -TD (bulk) and dt = +TD (inclusion)
%
% HISTORY
% A.A. Novotny     01/2013: code implementation
% A.A. Novotny     01/2018: code updating
%
%**************************************************************************

function dt = topder(U,psi,mesh,matprop)

    p = mesh.p; t = mesh.t; gamma = matprop.gamma; 
    tgamma = pdeintrp(p,t,(psi<0)+gamma*(psi>=0)); 
        
    la = matprop.la; mu = matprop.mu;
    alpha = (la+mu)/mu; beta = (la+3*mu)/(la+mu);
    [ux,uy] = pdegrad(p,t,U); id=[1 0 1]'; % solution gradient
    e = [ux(1,:);(ux(2,:)+uy(1,:))/2;uy(2,:)]; % strain
    divu = e(1,:) + e(3,:); s = la*id*divu + 2*mu*e; % nominal stress
    tgamma3 = [tgamma;tgamma;tgamma]; s = s.*tgamma3; % effective stress
            
    % topological derivative at the bulk phase
    coef0 = (1.0-gamma)/(1.0+gamma*beta); coef1 = coef0*(1.0+beta);
    coefx = (1.0-gamma)/(1.0+gamma*alpha); coef2 = 0.5*coef0*(alpha-beta)*coefx;
    
    dte = coef1*((s(1,:).*e(1,:)+2*s(2,:).*e(2,:)+s(3,:).*e(3,:))) ...
        + coef2*((s(1,:)+s(3,:)).*(e(1,:)+e(3,:))); 
                        
    % topological derivative at the inclusion
    gamma = 1.0/gamma;
    coef0 = (1.0-gamma)/(1.0+gamma*beta); coef1 = coef0*(1.0+beta);
    coefx = (1.0-gamma)/(1.0+gamma*alpha); coef2 = 0.5*coef0*(alpha-beta)*coefx;
      
    dti = coef1*((s(1,:).*e(1,:)+2*s(2,:).*e(2,:)+s(3,:).*e(3,:))) ...
        + coef2*((s(1,:)+s(3,:)).*(e(1,:)+e(3,:))); 
    
    % smoothing of the topological derivative
    pchi = (psi<0); tchi = pdeintrp(p,t,pchi);
   
    % compute function g: g = -DT (bulk) and g = DT (inclusion)
    dt = -tchi.*dte + (1-tchi).*dti; dt = pdeprtni(p,t,dt);
    %figure(7); clf; pdesurf(p,t,dt); pause
    
    
end