%**************************************************************************
% Structural Compliance Topology Optimization with Volume Constraint  
%**************************************************************************
% 
% DESCRIPTION
% Computes the topological derivative and use it together with a 
% level-set domain representation method in the context of structural 
% compliance topology optimization design
%
% HISTORY
% A.A. Novotny     01/2013: code implementation
% A.A. Novotny     02/2020: code updating
%
% MAIN REFERENCE
%
% A.A. Novotny & J. Sokolowski. Introduction to the Topological Derivative 
% Method, SpringerBriefs in Mathematics. Springer Nature, Switzerland, 2020 
% Chapter 5, Section 5.3.1, https://www.springer.com/gp/book/9783030369149
%
%**************************************************************************

clear all; close all; format long e; 

% load problem datas
mesh = []; psi = []; params = []; example = @hand;
% load problem data
cd('examples')
    [mesh, pdecoef, matprop, params, bc, psi] = example(mesh,psi,params);      
cd ..

% mesh and geometry  parameters
p = mesh.p; e = mesh.e; t = mesh.t; g = mesh.g; ghold = mesh.ghold; 
area = mesh.area; np = size(p,2); vol0 = sum(area);
% topology optimization parameters
stop = params.stop; kmin = params.kmin; penalty = params.penalty; 

% freeze nodes
phold = [];
for i = 1:size(mesh.ghold,1)
    phold = cat(1,phold,unique(t(1:3,t(4,:)==mesh.ghold(i))));
end
pfree = setdiff(1:size(p,2),phold); % free nodes

figure(1); clf; set(1,'WindowStyle','docked');
pdeplot(p,e,t,'xydata',t(4,:),'xystyle','flat','colormap','gray',...
              'xygrid','off','colorbar','off','title','Geometry'); 
axis image; axis off;

[~,unitM,~] = assema(p,t,0,1,0); % mass matrix of unity density
psi = psi/sqrt(dot(unitM*psi,psi)); % level-set function nomalization 
tchi = pdeintrp(p,t,(psi<0)); vol = dot(area,tchi); % volume

% plot initial guess
figure(2); clf; set(2,'WindowStyle','docked');
pdeplot(p,e,t,'xydata',(psi>0),'xystyle','flat','colormap','gray',...
              'xygrid','off','colorbar','off','title','Initial Guess'); 
axis image; axis off;

% hold-all domain
sux = matprop; aux.gamma = 1.0;
[U,F] = pdesolve(psi,mesh,aux,pdecoef,bc); 
comp0 = dot(F,U);

% solve linear system
[U,F] = pdesolve(psi,mesh,matprop,pdecoef,bc); 
energy = dot(F,U); 

% compute shape function
sf = energy/comp0 + penalty * vol/vol0;  

% plot deformed configuration
scale = 4.0/norm(U); Ud = scale*U'; 
pd(1,:) = p(1,:) + Ud(1:np); pd(2,:) = p(2,:) + Ud(np+1:2*np);

figure(10); clf; 
pdemesh(pd,e,t); axis image; axis off; 

k = 1; iter = 0; option = 'null'; 
gsf = sf; gth = pi; git = iter; gvo = vol;
while not(strcmp(option,'s'))
        
    iter = iter + 1; 
    if(iter == 200) 
        option = 's';
    end
    
    dt = topder(U,psi,mesh,matprop); 
    dt = dt/comp0 + penalty/vol0;
    dt = dt/sqrt(dot(unitM*dt,dt)); 
    dt(phold) = psi(phold); % freeze dt function
    cosin = max(min(dot(unitM*dt,psi),1.0),-1.0);
    theta = max(real(acos(cosin)),1.0e-4);

    % performs a line-search 
    sfold = sf; psiold = psi; energyold = energy;
    sf = sf + 1.0; k = min(1,1.5*k); % trick
 
    while (sf > sfold)

        if(k < kmin / 2)             
            psi = psiold; sf = sfold; energy = energyold;
            break;
        end  
        
        % update level-set function
        psi(pfree)  = (sin((1-k)*theta)*psiold(pfree)...
                    +  sin(k*theta)*dt(pfree))./sin(theta);
        
        psi = psi/sqrt(dot(unitM*psi,psi));  
        
        % solve linear system
        [U,F] = pdesolve(psi,mesh,matprop,pdecoef,bc); energy = dot(F,U);        
        % update the volume of the bulk phase
        tchi = pdeintrp(p,t,(psi < 0)); vol = dot(area,tchi);     
       
        % compute shape function
        sf = energy/comp0 + penalty * vol/vol0; 
                   
        k = k / 2;     

    end   
            
    k = k * 2; 

    git = [git,iter]; gsf = [gsf,sf]; gth = [gth,theta]; gvo = [gvo,vol];    
    
    disp(['iter   = ', num2str(iter)]);
    disp(['volume = ', num2str(vol),' => ', num2str(vol*100/vol0), '%']);
    disp(['energy = ', num2str(energy)]);
    disp(['shfunc = ', num2str(sf)]); 
    disp(['kappa  = ', num2str(k)]); 
    disp(['theta  = ', num2str(theta*180/pi)]);
    disp('%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%');
    
    drawnow
    figure(3); clf; set(3,'WindowStyle','docked');
    pdeplot(p,e,t,'xydata', (psi>0),'xystyle','flat','colormap','gray',...
                  'xygrid','off','colorbar','off','title','Topology'); 
    axis image; axis off;           
           
    figure(4); clf; set(4,'WindowStyle','docked');
    plot(git,gsf, ':*k'); title('Shape Function');
    
    figure(5); clf; set(5,'WindowStyle','docked');   
    plot(git,gth*180/pi, ':*k'); title('Theta Angle');
    
    figure(6); clf; set(6,'WindowStyle','docked');
    plot(git,gvo/vol0, ':*k'); title('Volume Fraction');   
    
    if or(k < kmin,theta < stop)
        % stop or try a mesh refinement        
        while and(not(strcmp(option,'r')), not(strcmp(option,'s')))
          option = input('\n -> type "r" to remesh or "s" to stop : ', 's');
        end
        
        if (option == 'r')
            cd('examples')
                [mesh, pdecoef, matprop, params, bc, psi] = example(mesh,psi,params);
            cd ..
            % update data            
            p = mesh.p; e = mesh.e; t = mesh.t; 
            np = size(p,2); area = mesh.area;   
            [~,unitM,~] = assema(p,t,0,1,0);
            psi = psi/sqrt(dot(unitM*psi,psi)); 
            tchi = pdeintrp(p,t,(psi < 0)); 
            vol = dot(area,tchi); % volume            
            aux = matprop; aux.gamma = 1.0; % hold-all domain
            [U,F] = pdesolve(psi,mesh,aux,pdecoef,bc); comp0 = dot(F,U);
            [U,F] = pdesolve(psi,mesh,matprop,pdecoef,bc); energy = dot(F,U); 
            sf = energy/comp0 + penalty * vol/vol0; k = 1;
            
            % freeze groups
            phold = [];
            for i = 1:size(mesh.ghold,1)
                phold = cat(1,phold,unique(t(1:3,t(4,:)==mesh.ghold(i)))); % nodes to block
            end
            pfree = setdiff(1:size(p,2),phold);
            option = 'null';
            
        end
    end
end

fig = input(' -> export figures? type "y" or "n"   : ', 's');
if strcmp(fig,'y')
    cd('results')
        print (1, '-dtiff', 'geometry');
        print (2, '-dtiff', 'iniguess');
        print (3, '-dtiff', 'topology');
        print (4, '-deps',  'shfunc');
        print (5, '-deps',  'theta');
        print (6, '-deps',  'volume');
    cd ..
else
    return;
end

%figure(7); clf; pdesurf(p,t,dt);
%figure(8);clf; pdesurf(p,t,psi);
