%**************************************************************************
% Solve the pdetool FEM linear system  
%**************************************************************************
%
% DESCRIPTION
% Assemble and solve the pdetool finite element linear system 
% 
% INPUT
% psi:     level-set function
% mesh:    pdetool mesh struct
% matprop: material properties struct
% pdecoef: pdetool coeficiets struct
% bc0:     standard boundary conditions struct
% bcio:    input/output boundary conditions struct
%
% OUTPUT
% [K,Fi,Fo]: pdetool stiffness matrix and input/output load vectors
%
% HISTORY
% A.A. Novotny     06/2014: code implementation
% A.A. Novotny     01/2018: code updating
%
%**************************************************************************

function [K,Fi,Fo] = pdesystem(psi, mesh, matprop, pdecoef, bc0, bcio)

    % mesh parameters
    p = mesh.p; t = mesh.t; np = size(p,2); 
    
    % material properties
    gamma = matprop.gamma; % contrast    
    tgamma = pdeintrp(p,t,(psi<0) + gamma*(psi>=0)); 

    % pde coefficients    
    f0 = zeros(2,1); dof = size(f0,1); a0 = pdecoef.a; c0 = pdecoef.c;
    
    % build stiffness metrix and load vector
    c = c0 * tgamma; % effective elasticity tensor 
    [K,M,~] = assema(p,t,c,a0,f0); K = K + M;

    % apply Dirichlet boundary condition
    eq = bc0.pDir(:,1) + (bc0.pDir(:,2)-1)*np; neq = size(eq,1);
    K(eq,:) = 0.0; K(:,eq) = 0.0; K(eq,eq) = speye(neq);

    % build input
    Fi = zeros(dof*np,1);
    eq = bcio.inp(:,1) + (bcio.inp(:,2)-1)*np; 
    Fi(eq) = bcio.inp(:,3); 

    % build output
    Fo = zeros(dof*np,1);
    eq = bcio.out(:,1) + (bcio.out(:,2)-1)*np; 
    Fo(eq) = bcio.out(:,3); 

end

