%**************************************************************************
% Topological Derivative for the Compliant Mechanism Shape Functinal  
%**************************************************************************
%
% DESCRIPTION
% Computes the topological derivative of the compliant mechanism
% 
% INPUT
% U:       direct solution
% V:       adjoint solution
% psi:     level-set function
% mesh:    pdetool mesh struct
% matprop: material properties struct
%
% OUTPUT
% dt = -TD (bulk) and dt = +TD (inclusion)
%
% HISTORY
% A.A. Novotny     06/2014: code implementation
% A.A. Novotny     01/2018: code updating
%
%**************************************************************************

function dt = topder(U,V,psi,mesh,matprop)

    p = mesh.p; t = mesh.t; gamma = matprop.gamma; 
    tgamma = pdeintrp(p,t,(psi<0)+gamma*(psi>=0)); 
        
    la = matprop.la; mu = matprop.mu; id=[1 0 1]';
    alpha = (la+mu)/mu; beta = (la+3*mu)/(la+mu);
    [ux,uy] = pdegrad(p,t,U); [vx,vy] = pdegrad(p,t,V); 
    ed = [ux(1,:);(ux(2,:)+uy(1,:))/2;uy(2,:)]; % direct strain
    ea = [vx(1,:);(vx(2,:)+vy(1,:))/2;vy(2,:)]; % adjoint strain
    divu = ed(1,:) + ed(3,:); sd = la*id*divu + 2*mu*ed; % nominal stress
    tgamma3 = [tgamma;tgamma;tgamma]; sd = sd.*tgamma3; % effective stress
            
    % topological derivative at the bulk phase
    coef0 = (1.0-gamma)/(1.0+gamma*beta); coef0 = -coef0; coef1 = coef0*(1.0+beta);
    coefx = (1.0-gamma)/(1.0+gamma*alpha); coef2 = 0.5*coef0*(alpha-beta)*coefx;
   
    dte = coef1*((sd(1,:).*ea(1,:)+2*sd(2,:).*ea(2,:)+sd(3,:).*ea(3,:))) ...
        + coef2*((sd(1,:)+sd(3,:)).*(ea(1,:)+ea(3,:)));   
                        
    % topological derivative at the inclusion
    gamma = 1.0/gamma;
    coef0 = (1.0-gamma)/(1.0+gamma*beta); coef0 = -coef0; coef1 = coef0*(1.0+beta);
    coefx = (1.0-gamma)/(1.0+gamma*alpha); coef2 = 0.5*coef0*(alpha-beta)*coefx;
      
    dti = coef1*((sd(1,:).*ea(1,:)+2*sd(2,:).*ea(2,:)+sd(3,:).*ea(3,:))) ...
        + coef2*((sd(1,:)+sd(3,:)).*(ea(1,:)+ea(3,:)));

    % smoothing of the topological derivative
    pchi = (psi<0); tchi = pdeintrp(p,t,pchi);
    % compute function g: g = -DT (bulk) and g = DT (inclusion)
    dt = -tchi.*dte + (1-tchi).*dti; dt = pdeprtni(p,t,dt);
    
end