%**************************************************************************
% Compliant Mechanisms Topology Design with Volume Constraint  
%**************************************************************************
% 
% DESCRIPTION
% Computes the topological derivative and use it together with a 
% level-set domain representation method in the context of compliant 
% mechanisms topology optimization design
%
% HISTORY
% A.A. Novotny     06/2014: code implementation
% A.A. Novotny     02/2020: code updating
%
% MAIN REFERENCE
%
% A.A. Novotny & J. Sokolowski. Introduction to the Topological Derivative 
% Method, SpringerBriefs in Mathematics. Springer Nature, Switzerland, 2020 
% Chapter 5, Section 5.3.2, https://www.springer.com/gp/book/9783030369149
%
%**************************************************************************

clear all; close all; format long e;

% load problem datas
mesh = []; psi = []; params = []; example = @invert;
% load problem data
cd('examples')
    [mesh, pdecoef, matprop, params, bc0, bcio, psi] = example(mesh,psi,params);      
cd ..

% mesh and geometry  parameters
p = mesh.p; e = mesh.e; t = mesh.t; g = mesh.g; 
area = mesh.area; np = size(p,2); vol0 = sum(area); 
% topology optimization parameters
stop = params.stop; kmin = params.kmin; 
penalty = params.penalty; alpha = params.alpha; 

[~,unitM,~] = assema(p,t,0,1,0); % mass matrix of unity density
psi = psi/sqrt(dot(unitM*psi,psi)); % level-set function nomalization 
tchi = pdeintrp(p,t,(psi < 0)); vol = dot(area,tchi); % volume

% plot initial guess
figure(1); clf; set(1,'WindowStyle','docked');
pdeplot(p,e,t,'xydata',(psi>0),'xystyle','flat','colormap','gray',...
              'xygrid','off','colorbar','off'); axis image; axis off;

% solve linear system
[K,Fi,Fo] = pdesystem(psi,mesh,matprop,pdecoef,bc0,bcio); 
Fd = Fi + Fo; U = K \ Fd;  % direct solution
Fa =-Fi - alpha*Fo; V = K \ Fa; % adjoint solution

% compute shape function
J = dot(Fi,U) + alpha*dot(Fo,U); J0 = J; 
sf = J/J0 + penalty * vol/vol0; 

% compute effectiveness
eff = -100*dot(Fo,U)/dot(Fi,U); 

k = 1; iter = 0; option = 'null'; 
gsf = sf; gth = pi; git = iter; gvo = vol; gef = eff;
while not(strcmp(option,'s'))
        
    iter = iter + 1; 
    if(iter == 200) 
        option = 's';
    end
    
    dt = topder(U,V,psi,mesh,matprop); dt = dt/J0 + penalty/vol0;
    dt = dt/sqrt(dot(unitM*dt,dt));  % g function normalization
            
    cosin = max(min(dot(unitM*dt,psi),1.0),-1.0);
    theta = max(real(acos(cosin)),1.0e-4);

    % performs a line-search 
    sfold = sf; psiold = psi; effold = eff; 
    sf = sf + 1.0; k = min(1,1.5*k); % trick
 
    while (sf > sfold)
        
        if(k < kmin / 2)             
            psi = psiold; sf = sfold; eff = effold; 
            break;
        end          

        % update level-set function
        psi = (sin((1-k)*theta)*psiold + sin(k*theta)*dt)./sin(theta);    
        psi = psi/sqrt(dot(unitM*psi,psi));  
        % solve linear system
        [K,Fi,Fo] = pdesystem(psi,mesh,matprop,pdecoef,bc0,bcio); 
        Fd = Fi + Fo; U = K \ Fd;  % direct solution
        J = dot(Fi,U) + alpha*dot(Fo,U);     
        % update the volume of the bulk phase
        tchi = pdeintrp(p,t,(psi < 0)); vol = dot(area,tchi);          
        % compute shape function
        sf = J/J0 + penalty * vol/vol0; 
        % compute effectiveness
        eff = -100*dot(Fo,U)/dot(Fi,U); 
        
        k = k / 2;   

    end   
            
    k = k * 2; 
    
    Fa =-Fi - alpha*Fo; V = K \ Fa; % adjoint solution

    git = [git,iter]; gsf = [gsf,sf]; gth = [gth,theta]; gef = [gef,eff]; gvo = [gvo,vol];    
    
    disp(['iter   = ', num2str(iter)]);
    disp(['volume = ', num2str(vol),' => ', num2str(vol*100/vol0), '%']);
    disp(['shfunc = ', num2str(sf)]); 
    disp(['eff    = ', num2str(eff), '%']); 
    disp(['kappa  = ', num2str(k)]); 
    disp(['theta  = ', num2str(theta*180/pi)]);
    disp('%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%');
    
    drawnow
    figure(2); clf; set(2,'WindowStyle','docked');
    pdeplot(p,e,t,'xydata',(psi>0),'xystyle','flat','colormap','gray',...
                  'xygrid','off','colorbar','off'); axis image; axis off;
              
    figure(3); clf; set(3,'WindowStyle','docked');
    plot(git,gsf, ':*k'); title('Shape Function');
    
    figure(4); clf; set(4,'WindowStyle','docked');   
    plot(git,gth*180/pi, ':*k'); title('Theta Angle');
    
    figure(5); clf; set(5,'WindowStyle','docked');
    plot(git,gvo/vol0, ':*k'); title('Volume Fraction');   
    
    figure(6); clf; set(6,'WindowStyle','docked');
    plot(gef, ':*k'); title('Effectiveness');   
    
    if or(k < kmin,theta < stop)
        % stop or try a mesh refinement        
        while and(not(strcmp(option,'r')), not(strcmp(option,'s')))
          option = input('\n -> type "r" to remesh or "s" to stop : ', 's');
        end
        
        if (option == 'r')
            cd('examples')
                [mesh, pdecoef, matprop, params, bc0, bcio, psi] = example(mesh,psi,params);
            cd ..
            % update data            
            p = mesh.p; e = mesh.e; t = mesh.t; 
            np = size(p,2); area = mesh.area;   
            [~,unitM,~] = assema(p,t,0,1,0);
            psi = psi/sqrt(dot(unitM*psi,psi)); 
            tchi = pdeintrp(p,t,(psi < 0)); 
            vol = dot(area,tchi); % volume
            [K,Fi,Fo] = pdesystem(psi,mesh,matprop,pdecoef,bc0,bcio);            
            Fd = Fi + Fo; U = K \ Fd;  % direct solution
            Fa =-Fi - alpha*Fo; V = K \ Fa; % adjoint solution
            J = dot(Fi,U) + alpha*dot(Fo,U); 
            sf = J/J0 + penalty * vol/vol0;
            option = 'null'; k = 1;
        end
    end
end

scale = 20/norm(U); Ud = scale*U'; 
pd(1,:) = p(1,:) + Ud(1:np); pd(2,:) = p(2,:) + Ud(np+1:2*np);
figure(10); clf;
pdeplot(pd,e,t,'xydata',(psi>0),'xystyle','flat','colormap','gray',...
               'xygrid','off','colorbar','off'); axis image; axis off;
           
fig = input(' -> export figures? "y" or "n"        : ', 's');
if strcmp(fig,'y')
    cd('results')
        print (1, '-dtiff', 'iniguess');
        print (2, '-dtiff', 'topology');
        print (3, '-deps',  'shfunc');
        print (4, '-deps',  'theta');
        print (5, '-deps',  'volume');
        print (6, '-deps',  'effective');
    cd ..
else
    return;
end

%figure(7); clf; pdesurf(p,t,dt);
%figure(8);clf; pdesurf(p,t,psi);
